/*************************************************************************************************
 *
 *   Copyright (c) Hilscher GmbH. All Rights Reserved.
 *
 **************************************************************************************************
 * @file Epl_Application.cpp
 *
 * @brief This file contains all functions of the Ethernet POWERLINK example application.
 * The example application is based upon Hilscher CifX Shared Memory Api.
 * 
 * @author Elmar
 *
 */

#include "stdafx.h"
#include "Debug.h"
#include "Common.h"
#include "Epl_Application.h"


/*************************************************************************************************
 * @brief This function creates a simple variable object (no subobjects).
 * 
 * @param ptEplApp Pointer to Epl application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT Epl_CreateSimpleObject(EPL_APPLICATION_T* ptEplApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptEplApp;

  /* create object 0x6000 within object dictionary */
  /* variable, UINT32, no read / write restrictions, 1 subobject */
  EPLCN_PCK_OD_CREATE_OBJECT_REQ_T* ptCreateObjReq = (EPLCN_PCK_OD_CREATE_OBJECT_REQ_T*)&tSendPkt;
  memset(ptCreateObjReq, 0, sizeof(EPLCN_PCK_OD_CREATE_OBJECT_REQ));

  /* fill packet header */
  ptCreateObjReq->tHead.ulDest            = 0x20;
  ptCreateObjReq->tHead.ulSrc             = 0x00;
  ptCreateObjReq->tHead.ulDestId          = 0x00;
  ptCreateObjReq->tHead.ulSrcId           = 0x00;
  ptCreateObjReq->tHead.ulLen             = 0x09;
  ptCreateObjReq->tHead.ulId              = 0x00;
  ptCreateObjReq->tHead.ulSta             = 0x00;
  ptCreateObjReq->tHead.ulCmd             = EPLCN_PCK_OD_CREATE_OBJECT_REQ;
  ptCreateObjReq->tHead.ulExt             = 0x00;
  ptCreateObjReq->tHead.ulRout            = 0x00;

  /* fill data part of packet */
  ptCreateObjReq->tData.usIndex           = 0x6000;
  ptCreateObjReq->tData.bNumOfSubObjs     = 0x01; /* we must have at least one sub-object which contains our data */
  ptCreateObjReq->tData.bMaxNumOfSubObjs  = 0x01;
  ptCreateObjReq->tData.usObjAccess       = 0x00; 
  ptCreateObjReq->tData.bObjectCode       = 0x07; /* object code for variable, see Powerlink CN API manual */
  ptCreateObjReq->tData.usDatatype        = 0x07; /* object code for UINT32, see Powerlink CN API manual */

  /* fire the request */
  tResult = SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* at least one subobject is needed */
  EPLCN_PCK_OD_CREATE_SUBOBJECT_REQ_T* ptCreateSubObjReq = (EPLCN_PCK_OD_CREATE_SUBOBJECT_REQ_T*)&tSendPkt;
  memset(ptCreateSubObjReq, 0, sizeof(EPLCN_PCK_OD_CREATE_SUBOBJECT_REQ_T));

  /* fill packet header */
  ptCreateSubObjReq->tHead.ulDest            = 0x20;
  ptCreateSubObjReq->tHead.ulSrc             = 0x00;
  ptCreateSubObjReq->tHead.ulDestId          = 0x00;
  ptCreateSubObjReq->tHead.ulSrcId           = 0x00;
  ptCreateSubObjReq->tHead.ulLen             = 19;
  ptCreateSubObjReq->tHead.ulId              = 0x00;
  ptCreateSubObjReq->tHead.ulSta             = 0x00;
  ptCreateSubObjReq->tHead.ulCmd             = EPLCN_PCK_OD_CREATE_SUBOBJECT_REQ;
  ptCreateSubObjReq->tHead.ulExt             = 0x00;
  ptCreateSubObjReq->tHead.ulRout            = 0x00;

  /* fill data part of packet */
  ptCreateSubObjReq->tData.ulMode            = 0x00;   /* stored in a selfcontained memory area */
  ptCreateSubObjReq->tData.usIndex           = 0x6000;
  ptCreateSubObjReq->tData.bSubIdx           = 1;
  ptCreateSubObjReq->tData.usDirection       = EPLCN_PCK_OD_SUBOBJECT_DIRECTION_NOT_DEF;   /* direction not defined */
  ptCreateSubObjReq->tData.usSubObjAccess    = 0xFFFF; /* rw access in al states */
  ptCreateSubObjReq->tData.usDatatype        = 0x07;
  ptCreateSubObjReq->tData.usFieldLen        = 0x01;   /* fixed to 1 see api manual*/
  ptCreateSubObjReq->tData.ulRelativeAddress = 0x00;

  /* fire the request */
  tResult = SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* set initial value of object */
  EPLCN_PCK_OD_WRITE_OBJECT_REQ_T* ptOdWriteReq = (EPLCN_PCK_OD_WRITE_OBJECT_REQ_T*)&tSendPkt;
  memset(ptOdWriteReq, 0, sizeof(EPLCN_PCK_OD_WRITE_OBJECT_REQ_T));

  /* fill packet header */
  ptOdWriteReq->tHead.ulDest            = 0x20;
  ptOdWriteReq->tHead.ulSrc             = 0x00;
  ptOdWriteReq->tHead.ulDestId          = 0x00;
  ptOdWriteReq->tHead.ulSrcId           = 0x00;
  ptOdWriteReq->tHead.ulLen             = 3+4;
  ptOdWriteReq->tHead.ulId              = 0x00;
  ptOdWriteReq->tHead.ulSta             = 0x00;
  ptOdWriteReq->tHead.ulCmd             = EPLCN_PCK_OD_WRITE_OBJECT_REQ;
  ptOdWriteReq->tHead.ulExt             = 0x00;
  ptOdWriteReq->tHead.ulRout            = 0x00;

  /* fill data part of packet */   
  ptOdWriteReq->tData.usIndex           = 0x6000;
  ptOdWriteReq->tData.bSubIdx           = 0x01;  

  /* data follows afterwards */
  TLR_UINT8* pbData = (TLR_UINT8*)(&ptOdWriteReq->tData+1);
  pbData[0]               = 0xFF;
  pbData[1]               = 0x00;
  pbData[2]               = 0xAA;
  pbData[3]               = 0x00;

  /* fire the request */
  tResult = SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* naming of object makes no sense, */
  /* because name is not accessible form external (via bus) */

  return tResult;
}



/*************************************************************************************************
 * @brief This function creates all objects within the Epl stack.
 * 
 * @param ptEplApp Pointer to Epl application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT CreateObjects(EPL_APPLICATION_T* ptEplApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptEplApp;

  /* check first for some other packets to receive / empty the queue */
  while(TLR_S_OK == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(tRecvPkt), &tRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    HandlePacket((APPLICATION_T*)ptApp, &tRecvPkt);
  }

  /* creating a simple variable object */
  tResult = Epl_CreateSimpleObject(ptEplApp);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}



/*************************************************************************************************
 * @brief This function registers for all necessary sdo notify indications.
 * 
 * @param ptEplApp Pointer to Epl application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT Epl_RegisterNotifications(EPL_APPLICATION_T* ptEplApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;
  
  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptEplApp;

  /* we register for object 0x6000 write notifications */
  EPLCN_PCK_OD_NOTIFY_REGISTER_REQ_T* ptNotifyReq = (EPLCN_PCK_OD_NOTIFY_REGISTER_REQ_T*)&tSendPkt;

  /* fill packet header */
  ptNotifyReq->tHead.ulDest        = 0x20;
  ptNotifyReq->tHead.ulSrc         = 0x00;
  ptNotifyReq->tHead.ulDestId      = 0x00;
  ptNotifyReq->tHead.ulSrcId       = 0x00;
  ptNotifyReq->tHead.ulLen         = 10;
  ptNotifyReq->tHead.ulId          = 0x00;
  ptNotifyReq->tHead.ulSta         = 0x00;
  ptNotifyReq->tHead.ulCmd         = EPLCN_PCK_OD_NOTIFY_REGISTER_REQ;
  ptNotifyReq->tHead.ulExt         = 0x00;
  ptNotifyReq->tHead.ulRout        = 0x00;

  /* fill data part of packet */
  ptNotifyReq->tData.usIndex       = 0x6000;
  ptNotifyReq->tData.fReadNotify   = 0x00;
  ptNotifyReq->tData.fWriteNotify  = 0x01;

  tResult = SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}


/*************************************************************************************************
 * @brief This function unregisters from sdo notify indications.
 * 
 * @param ptEplApp Pointer to Epl application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT Epl_UnegisterNotifications(EPL_APPLICATION_T* ptEplApp)
{  
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptEplApp;

  /* we umregister from object 0x6000 write notifications */
  EPLCN_PCK_OD_NOTIFY_UNREGISTER_REQ_T* ptUnregReq = (EPLCN_PCK_OD_NOTIFY_UNREGISTER_REQ_T*)&tSendPkt;

  /* fill packet header */
  ptUnregReq->tHead.ulDest        = 0x20;
  ptUnregReq->tHead.ulSrc         = 0x00;
  ptUnregReq->tHead.ulDestId      = 0x00;
  ptUnregReq->tHead.ulSrcId       = 0x00;
  ptUnregReq->tHead.ulLen         = 10;
  ptUnregReq->tHead.ulId          = 0x00;
  ptUnregReq->tHead.ulSta         = 0x00;
  ptUnregReq->tHead.ulCmd         = EPLCN_PCK_OD_NOTIFY_UNREGISTER_REQ;
  ptUnregReq->tHead.ulExt         = 0x00;
  ptUnregReq->tHead.ulRout        = 0x00;

  /* fill data part of packet */
  ptUnregReq->tData.usIndex       = 0x6000;
  ptUnregReq->tData.fReadNotify   = TLR_FALSE;
  ptUnregReq->tData.fWriteNotify  = TLR_FALSE;

  /* fire the request */
  tResult = SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}


/*************************************************************************************************
 * @brief This function handles all sdo write indications.
 * 
 * @param ptEplApp          Pointer to Epl application data.
 * @param ptPacket          Pointer to CifX packet.
 *
 */
static void Epl_HandleWriteIndication(EPL_APPLICATION_T* ptEplApp, EPLCN_PCK_OD_NOTIFY_WRITE_IND_T* ptPacket)
{
  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptEplApp;

  TLR_UINT32* ptValue = NULL;

  switch(ptPacket->tData.usObjIndex)
  {
    /********************************
     * Sdo 0x6000:  Demo Object
     */    
    case 0x6000:
      /* prompt some information */
      ptValue = (TLR_UINT32*)(&ptPacket->tData + 1);
      DEBUG("\n");
      DEBUG("Object 0x6000 has been written.\n");
      DEBUG("New Value: 0x%x\n", *ptValue );

      switch(*ptValue)
      {
        case 0x5555:
          { /* A non-critical emergency does NOT trigger a state change to PreOp1 */
            /* REMARKS: The packet EPLCN_PCK_SEND_EMERGENCY_REQ requires a device with a specified profile in device type */
            EPLCN_PCK_SEND_EMERGENCY_REQ_T tNonCriticalEmergency;
            tNonCriticalEmergency.tHead.ulCmd = EPLCN_PCK_SEND_EMERGENCY_REQ;
            tNonCriticalEmergency.tHead.ulSta = 0;
            tNonCriticalEmergency.tHead.ulDest = 0x20;
            tNonCriticalEmergency.tHead.ulDestId = 0;
            tNonCriticalEmergency.tHead.ulSrc = 0;
            tNonCriticalEmergency.tHead.ulSrcId = 0;
            tNonCriticalEmergency.tHead.ulExt = 0;
            tNonCriticalEmergency.tHead.ulRout = 0;
            tNonCriticalEmergency.tHead.ulLen = sizeof(tNonCriticalEmergency.tData);
            tNonCriticalEmergency.tHead.ulId = 0;
            tNonCriticalEmergency.tData.usErrorCode = 0x5555;
            tNonCriticalEmergency.tData.bErrorRegister = 0;
            tNonCriticalEmergency.tData.abManufSpecific[0] = 0x55;
            tNonCriticalEmergency.tData.abManufSpecific[1] = 0x55;
            tNonCriticalEmergency.tData.abManufSpecific[2] = 0x55;
            tNonCriticalEmergency.tData.abManufSpecific[3] = 0x55;
            tNonCriticalEmergency.tData.abManufSpecific[4] = 0x55;
            tNonCriticalEmergency.tData.fEnterErrorCondition = TLR_FALSE;

            xChannelPutPacket(ptApp->tHead.hChannel, (CIFX_PACKET*)&tNonCriticalEmergency, ptApp->tHead.ulTimeout);
          }
          break;

        case 0x5556:
          { /* A critical emergency DOES trigger a state change to PreOp1 */
            /* REMARKS: The packet EPLCN_PCK_SEND_EMERGENCY_REQ requires a device with a specified profile in device type */
            EPLCN_PCK_SEND_EMERGENCY_REQ_T tCriticalEmergency;
            tCriticalEmergency.tHead.ulCmd = EPLCN_PCK_SEND_EMERGENCY_REQ;
            tCriticalEmergency.tHead.ulSta = 0;
            tCriticalEmergency.tHead.ulDest = 0x20;
            tCriticalEmergency.tHead.ulDestId = 0;
            tCriticalEmergency.tHead.ulSrc = 0;
            tCriticalEmergency.tHead.ulSrcId = 0;
            tCriticalEmergency.tHead.ulExt = 0;
            tCriticalEmergency.tHead.ulRout = 0;
            tCriticalEmergency.tHead.ulLen = sizeof(tCriticalEmergency.tData);
            tCriticalEmergency.tHead.ulId = 0;
            tCriticalEmergency.tData.usErrorCode = 0x5556;
            tCriticalEmergency.tData.bErrorRegister = 0;
            tCriticalEmergency.tData.abManufSpecific[0] = 0x56;
            tCriticalEmergency.tData.abManufSpecific[1] = 0x56;
            tCriticalEmergency.tData.abManufSpecific[2] = 0x56;
            tCriticalEmergency.tData.abManufSpecific[3] = 0x56;
            tCriticalEmergency.tData.abManufSpecific[4] = 0x56;
            tCriticalEmergency.tData.fEnterErrorCondition = TLR_TRUE;

            xChannelPutPacket(ptApp->tHead.hChannel, (CIFX_PACKET*)&tCriticalEmergency, ptApp->tHead.ulTimeout);
          }
          break;

        case 0x5557:
          { /* A non-critical emergency does NOT trigger a state change to PreOp1 */
            /* Error Entry as defined by POWERLINK specification */
            EPLCN_PCK_WRITE_ERROR_ENTRY_REQ_T tNonCriticalEmergency;
            tNonCriticalEmergency.tHead.ulCmd = EPLCN_PCK_WRITE_ERROR_ENTRY_REQ;
            tNonCriticalEmergency.tHead.ulSta = 0;
            tNonCriticalEmergency.tHead.ulDest = 0x20;
            tNonCriticalEmergency.tHead.ulDestId = 0;
            tNonCriticalEmergency.tHead.ulSrc = 0;
            tNonCriticalEmergency.tHead.ulSrcId = 0;
            tNonCriticalEmergency.tHead.ulExt = 0;
            tNonCriticalEmergency.tHead.ulRout = 0;
            tNonCriticalEmergency.tHead.ulLen = sizeof(tNonCriticalEmergency.tData);
            tNonCriticalEmergency.tHead.ulId = 0;
            tNonCriticalEmergency.tData.usEntryType = 0x7000;
            tNonCriticalEmergency.tData.usErrorCode = 0x5557;
            tNonCriticalEmergency.tData.aulAddInformation[0] = 0x55555555;
            tNonCriticalEmergency.tData.aulAddInformation[1] = 0x55555555;
            tNonCriticalEmergency.tData.fEnterErrorCondition = TLR_FALSE;

            xChannelPutPacket(ptApp->tHead.hChannel, (CIFX_PACKET*)&tNonCriticalEmergency, ptApp->tHead.ulTimeout);
          }
          break;

        case 0x5558:
          { /* A critical emergency DOES trigger a state change to PreOp1 */
            /* Error Entry as defined by POWERLINK specification */
            EPLCN_PCK_WRITE_ERROR_ENTRY_REQ_T tCriticalEmergency;
            tCriticalEmergency.tHead.ulCmd = EPLCN_PCK_WRITE_ERROR_ENTRY_REQ;
            tCriticalEmergency.tHead.ulSta = 0;
            tCriticalEmergency.tHead.ulDest = 0x20;
            tCriticalEmergency.tHead.ulDestId = 0;
            tCriticalEmergency.tHead.ulSrc = 0;
            tCriticalEmergency.tHead.ulSrcId = 0;
            tCriticalEmergency.tHead.ulExt = 0;
            tCriticalEmergency.tHead.ulRout = 0;
            tCriticalEmergency.tHead.ulLen = sizeof(tCriticalEmergency.tData);
            tCriticalEmergency.tHead.ulId = 0;
            tCriticalEmergency.tData.usEntryType = 0x7000;
            tCriticalEmergency.tData.usErrorCode = 0x5558;
            tCriticalEmergency.tData.aulAddInformation[0] = 0x55555555;
            tCriticalEmergency.tData.aulAddInformation[1] = 0x55555555;
            tCriticalEmergency.tData.fEnterErrorCondition = TLR_TRUE;

            xChannelPutPacket(ptApp->tHead.hChannel, (CIFX_PACKET*)&tCriticalEmergency, ptApp->tHead.ulTimeout);
          }
          break;

        case 0x5559:
          { /* CANopen emergency mapped by hand with EPLCN_PCK_WRITE_ERROR_ENTRY_REQ */
            EPLCN_PCK_WRITE_ERROR_ENTRY_REQ_T tCriticalEmergency;
            tCriticalEmergency.tHead.ulCmd = EPLCN_PCK_WRITE_ERROR_ENTRY_REQ;
            tCriticalEmergency.tHead.ulSta = 0;
            tCriticalEmergency.tHead.ulDest = 0x20;
            tCriticalEmergency.tHead.ulDestId = 0;
            tCriticalEmergency.tHead.ulSrc = 0;
            tCriticalEmergency.tHead.ulSrcId = 0;
            tCriticalEmergency.tHead.ulExt = 0;
            tCriticalEmergency.tHead.ulRout = 0;
            tCriticalEmergency.tHead.ulLen = sizeof(tCriticalEmergency.tData);
            tCriticalEmergency.tHead.ulId = 0;
            tCriticalEmergency.tData.usEntryType = 0x7401;  /* lower 12 bit correspond to profile number in device type (in this case DS401 is used) */
                                                            /* the upper 4 bit must be set to 0x7 */
            tCriticalEmergency.tData.usErrorCode = 0x6000;  /* ErrorCode is identical to CANopen emergency error code */
            tCriticalEmergency.tData.aulAddInformation[0] = 0x55555555;  /* bytes 0-3 of manufacturer specific of CANopen emergency */
            tCriticalEmergency.tData.aulAddInformation[1] = 0x00005555;  /* bytes 4-5 of manufacturer specific of CANopen emergency */
            tCriticalEmergency.tData.fEnterErrorCondition = TLR_FALSE;

            xChannelPutPacket(ptApp->tHead.hChannel, (CIFX_PACKET*)&tCriticalEmergency, ptApp->tHead.ulTimeout);
          }
          break;

      }

      /* build the response and send it */
      ptPacket->tHead.ulCmd |= 0x01;
      xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET*)ptPacket, ptApp->tCommon.ulTimeout);
      break;

    default:
      /* build the response and send it */
      ptPacket->tHead.ulSta = TLR_E_OD2_INVALID_OBJECT;
      ptPacket->tHead.ulCmd |= 0x01;

      xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET*)ptPacket, ptApp->tCommon.ulTimeout);
      break;
  }
}


/*************************************************************************************************
 * @brief This function handles all sdo read indications.
 * 
 * @param ptEplApp          Pointer to Epl application data.
 * @param ptPacket          Pointer to CifX packet.
 *
 */            
static void Epl_HandleReadIndication(EPL_APPLICATION_T* ptEplApp, EPLCN_PCK_OD_NOTIFY_READ_IND_T* ptPacket)
{
  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptEplApp;

  /* we expect no read indications */
  switch(ptPacket->tData.usObjIndex)
  {
    case 0x6000:
      /* implement handling here if necessary */
      break;
  
    default:
      /* build the response and send it */
      ptPacket->tHead.ulSta = TLR_E_OD2_INVALID_OBJECT;
      ptPacket->tHead.ulCmd |= 0x01;

      xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET*)ptPacket, ptApp->tCommon.ulTimeout);
      break;
  }
}




/*************************************************************************************************
 * @brief This method builds a configuration request packet.
 * 
 * @param CIFX_PACKET* ptPacket pointer to a CIFX_PACKET structure.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT BuildConfigurationReq (CIFX_PACKET* ptPacket)
{
  TLR_RESULT tResult = TLR_S_OK;

  EPLCN_DPM_WARMSTART_REQ_T *ptWarmstartReq = (EPLCN_DPM_WARMSTART_REQ_T*)ptPacket;

  ptWarmstartReq->tHead.ulDest                    = 0x20;                         /* Destination of packet                    */
  ptWarmstartReq->tHead.ulSrc                     = 0x10;                         /* Source of packet, process queue          */
  ptWarmstartReq->tHead.ulLen                     = sizeof(ptWarmstartReq->tData);/* Length of packet data without header     */
  ptWarmstartReq->tHead.ulCmd                     = EPLCN_DPM_WARMSTART_REQ;      /* Packet command                           */
  ptWarmstartReq->tHead.ulSta                     = 0;                            /* Status code of operation                 */
  ptWarmstartReq->tHead.ulExt                     = 0;                            /* extension                                */

  ptWarmstartReq->tData.ulSystemFlags             = 0;                            /* System Flags                             */
  ptWarmstartReq->tData.ulWatchdogTime            = 1000;                         /* Watchdog time                            */
  ptWarmstartReq->tData.ulVendorId                = 0x44;                         /* Vendor Id                                */

  /* TODO: modify*/
  ptWarmstartReq->tData.ulProductCode             = 0x00000019;                   /* Product code of CifX example application */
  ptWarmstartReq->tData.ulRevisionNumber          = 0;                            /* Revision number                          */
  ptWarmstartReq->tData.ulSerialNumber            = 0;                            /* Serial number                            */
  ptWarmstartReq->tData.ulStackConfigurationFlags = MSK_EPLCN_DPM_WARMSTART_STACK_CFG_CONFIGURE_DEFAULT_OBJECTS |
                                                    MSK_EPLCN_DPM_WARMSTART_STACK_CFG_DISABLE_PDO_MAP_VERS_CHECK;
                                                                                  /* Configuration Flags                      */
  ptWarmstartReq->tData.ulThresholdDisableFlags   = 0;                            /* Threshold Disable Flags                  */
  ptWarmstartReq->tData.ulThresholdLossSoC        = 0;                            /* LossSoC Threshold   (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdLossPReq       = 0;                            /* LossPReq Threshold  (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdLossSoA        = 0;                            /* LossSoA Threshold   (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdSoCJitter      = 0;                            /* SoCJitter Threshold (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdCollision      = 0;                            /* Collision Threshold (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdCrcError       = 0;                            /* CrcError Threshold  (ignored if set to 0)*/
  ptWarmstartReq->tData.ulCycleLength             = 1000;                         /* Cycle Length configuration (ns)          */
  ptWarmstartReq->tData.ulSoCJitterRange          = 2000;                         /* SoC Jitter range configuration (ns)      */
  ptWarmstartReq->tData.usProcessDataOutputSize   = MAX_RTD_SIZE;                 /* Process Data Output Size                 */
  ptWarmstartReq->tData.usProcessDataInputSize    = MAX_RTD_SIZE;                 /* Process Data Input Size                  */
  memcpy(ptWarmstartReq->tData.abNodeName, "CIFX RE EXAMPLE PLS", sizeof(ptWarmstartReq->tData.abNodeName));
                                                                                  /* EPL node name                            */
  ptWarmstartReq->tData.ulGatewayAddress          = 0xC0A864FE;                   /* IP Gateway: 192.168.100.254              */

  ptWarmstartReq->tData.bNodeId                   = 1;                            /* EPL node id (range 1 to 239)             */

  ptWarmstartReq->tData.bSoCTriggerConfig         = 0;                            /* SoC Trigger configuration                */
  ptWarmstartReq->tData.ulSoCTriggerDelay         = 0;                            /* SoC Trigger Delay configuration          */
  ptWarmstartReq->tData.ulSoCTriggerLength        = 0;                            /* SoC Trigger Length configuration         */
  ptWarmstartReq->tData.bPReqMappingVersion       = 0;                            /* PReq Mapping Version                     */
  ptWarmstartReq->tData.bPResMappingVersion       = 0;                            /* PRes Mapping Version                     */
  ptWarmstartReq->tData.bNumberOfStatusEntries    = 0;                            /* configured status entries                */

  return tResult;
}


/*************************************************************************************************
 * @brief This function initializes the application. 
 * It allocates memory for the application data and returns a pointer to it.
 * App_Finalize() must be called in order to achieve a friendly shutdown and to release memory again.
 * 
 * @param pptApp    Pointer to Pointer to application data.
 * @param hDriver   Driver Handle.
 * @param hChannel  Channel Handle.
 * @param ulTimeout Standard timeout for packet operations.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Initialize(APPLICATION_T** pptApp, TLR_HANDLE hDriver, TLR_HANDLE hChannel, TLR_UINT32 ulTimeout)
{
  DEBUG("Initializing application...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* at the end of the function we will return the pointer */
  *pptApp = NULL;

  DEBUG("Allocating memory...\n");

  /* allocate memory for application data */
  EPL_APPLICATION_T* ptEplApp = (EPL_APPLICATION_T*)malloc(sizeof(EPL_APPLICATION_T));
  
  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptEplApp;

  /* fill application standard header */
  ptApp->tCommon.hDriver   = hDriver;
  ptApp->tCommon.hChannel  = hChannel;
  ptApp->tCommon.ulTimeout = ulTimeout;

  /* allocate memory for read / write buffers */
  ptApp->tCommon.ulReadBufferSize  = MAX_RTD_SIZE * sizeof(TLR_UINT8);
  ptApp->tCommon.ulWriteBufferSize = MAX_RTD_SIZE * sizeof(TLR_UINT8);

  ptApp->tCommon.pabReadBuffer  =  (TLR_UINT8*) malloc(ptApp->tCommon.ulReadBufferSize);
  ptApp->tCommon.pabWriteBuffer =  (TLR_UINT8*) malloc(ptApp->tCommon.ulWriteBufferSize);

  /* initialize the read and write buffer with zero */
  memset(ptApp->tCommon.pabReadBuffer,  0, ptApp->tCommon.ulReadBufferSize);
  memset(ptApp->tCommon.pabWriteBuffer, 0, ptApp->tCommon.ulWriteBufferSize);

  DEBUG("Successful.\n");

  /* register the application at the stack */
  tResult = SendRecvEmptyPkt(ptApp, RCX_REGISTER_APP_REQ);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* create all user objects */
  tResult = CreateObjects(ptEplApp);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* register read / write notifications */
  tResult = Epl_RegisterNotifications(ptEplApp);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* return the pointer */
  *pptApp = ptApp;

  return tResult;
}


/*************************************************************************************************
 * @brief This method finalizes the EPL application. 
 * It returns handles, frees memory, aso.
 * 
 * @param ptApp Pointer to Epl application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Finalize(APPLICATION_T* ptApp)
{
  DEBUG("Shutdown application...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to specific application */
  EPL_APPLICATION_T* ptEplApp = (EPL_APPLICATION_T*)ptApp;

  /* at first we empty the queue, than we start with the shutdown sequence below */
  while(TLR_S_OK == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(tRecvPkt), &tRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    HandlePacket(ptApp, &tRecvPkt);
  }

  /* unregister from read / write notifications */
  Epl_UnegisterNotifications(ptEplApp);

  /* unregister on EcatEsm task (indications for status) */
  SendRecvEmptyPkt(ptApp, RCX_UNREGISTER_APP_REQ);


  DEBUG("Free Resources...\n");

  /* free buffer resources */
  free(ptApp->tCommon.pabReadBuffer);
  free(ptApp->tCommon.pabWriteBuffer);

  /* free application data container */
  free(ptApp);

  return tResult;
}



/*************************************************************************************************
 * @brief This method handles all packets which will be received by Epl application.
 * 
 * @param ptApp Pointer to application data.
 * @param ptPacket Pointer to CifX packet.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void HandlePacket(APPLICATION_T*  ptApp, CIFX_PACKET* ptPacket)
{
  /* convert to specific application */
  EPL_APPLICATION_T* ptEplApp = (EPL_APPLICATION_T*)ptApp;

  TLR_PACKET_HEADER_T* ptPck = (TLR_PACKET_HEADER_T*)ptPacket;

   switch(ptPck->ulCmd)
  {
    case EPLCN_PCK_OD_NOTIFY_WRITE_IND:
      Epl_HandleWriteIndication(ptEplApp, (EPLCN_PCK_OD_NOTIFY_WRITE_IND_T*)ptPacket);
      break;

    case EPLCN_PCK_OD_NOTIFY_READ_IND:
      Epl_HandleReadIndication(ptEplApp, (EPLCN_PCK_OD_NOTIFY_READ_IND_T*)ptPacket);
      break;

    default:
      ptPck->ulSta = TLR_E_UNKNOWN_COMMAND;
      ptPck->ulLen = 0;
      break;
  }
}


/*************************************************************************************************
 * @brief This method handles the cyclic process data exchange.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void HandleProcessData(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* convert to specific application */
  EPL_APPLICATION_T* ptEplApp = (EPL_APPLICATION_T*)ptApp;

  /* cyclic io handing happens here */
  /* read RTD-Data */
  tResult = xChannelIORead(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulReadBufferSize, ptApp->tCommon.pabReadBuffer, 2);

  /* the action depends on the return value we have received */
  if ((CIFX_DEV_EXCHANGE_FAILED == tResult) || (CIFX_DEV_EXCHANGE_TIMEOUT == tResult))
  {
    DEBUG("F");
  }
  else if (CIFX_DEV_NO_COM_FLAG == tResult)
  {
    DEBUG("F");
  }
  else if (CIFX_NO_ERROR == tResult)
  {
    DEBUG("X");
    
    /* we copy the memory we have read to the memory we want to send, */
    /* because we just want to mirror the data */
    memcpy (ptApp->tCommon.pabWriteBuffer, ptApp->tCommon.pabReadBuffer, ptApp->tCommon.ulWriteBufferSize);     

    // write RTD-Data 
    tResult = xChannelIOWrite(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulWriteBufferSize, ptApp->tCommon.pabWriteBuffer, 2);       
  }
  else
  {
    /* received unexpected failure */
    DEBUG("F");
  }
    
}









